# ABSTRACT: An event emitted by a Net::Object::Peer node
package Net::Object::Peer::Event;

use 5.10.0;

use Types::Standard qw[ ConsumerOf InstanceOf ];

use Moo;
use strictures 2;
use namespace::clean;

our $VERSION = '0.06'; # TRIAL

extends 'Beam::Event';

has '+emitter' => (
    is       => 'ro',
    isa      => ConsumerOf ['Net::Object::Peer'],
    required => 1,
);

#pod =attr  addr
#pod
#pod The reference address of the true emitter
#pod
#pod =cut

has addr => (
    is       => 'rwp',
    isa      => InstanceOf['Net::Object::Peer::RefAddr'],
    predicate => 1,
);

#pod =begin pod_coverage
#pod
#pod =head4 BUILD
#pod
#pod =head4 has_addr
#pod
#pod =end pod_coverage
#pod
#pod =cut

sub BUILD {

    # do this as soon as possible.  if it's lazy, emitter may disappear
    # before a lazy builder can run

    $_[0]->_set_addr( Net::Object::Peer::RefAddr->new( $_[0]->emitter ) )
      unless $_[0]->has_addr;
}

1;

#
# This file is part of Net-Object-Peer
#
# This software is Copyright (c) 2016 by Smithsonian Astrophysical Observatory.
#
# This is free software, licensed under:
#
#   The GNU General Public License, Version 3, June 2007
#

=pod

=head1 NAME

Net::Object::Peer::Event - An event emitted by a Net::Object::Peer node

=head1 VERSION

version 0.06

=head1 DESCRIPTION

B<Net::Object::Peer::Event> is a sub-class of L<Beam::Event>,
which adds

=over

=item *

the requirement that the emitter be a consumer of L<Net::Object::Peer>.

=item *

a new attribute, C<addr>, which will contain the refaddr of the true
emitter object, in the case that C<emitter> is masqueraded.

=back

Event classes used with L<Net::Object::Peer> must be derived from
this class.

=head1 ATTRIBUTES

=head2 addr

The reference address of the true emitter

=begin pod_coverage

=head4 BUILD

=head4 has_addr

=end pod_coverage

=head1 AUTHOR

Diab Jerius <djerius@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2016 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut

__END__

#pod =head1 DESCRIPTION
#pod
#pod B<Net::Object::Peer::Event> is a sub-class of L<Beam::Event>,
#pod which adds
#pod
#pod =over
#pod
#pod =item *
#pod
#pod the requirement that the emitter be a consumer of L<Net::Object::Peer>.
#pod
#pod =item *
#pod
#pod a new attribute, C<addr>, which will contain the refaddr of the true
#pod emitter object, in the case that C<emitter> is masqueraded.
#pod
#pod =back
#pod
#pod Event classes used with L<Net::Object::Peer> must be derived from
#pod this class.
