package Run::Parts::Debian;

# ABSTRACT: Perl interface to Debian's run-parts tool

use Modern::Perl;
use Run::Parts::Common;

our $VERSION = '0.07'; # VERSION generated by DZP::OurPkgVersion


sub new {
    my $self = {};
    bless($self, shift);
    $self->{dir} = shift;

    return $self;
}


sub run_parts_command {
    my $self = shift;
    my $rp_cmd = shift;

    my $command =
        "/bin/run-parts " .
        ($rp_cmd ? "'--$rp_cmd'" : '') .
        " '".$self->{dir}."'";

    return chomped_lines(`$command`);
}


q<debian/rules>; # End of Run::Parts::Debian

__END__

=pod

=encoding UTF-8

=head1 NAME

Run::Parts::Debian - Perl interface to Debian's run-parts tool

=head1 VERSION

version 0.07

=head1 SYNOPSIS

Perl interface to Debian's L<run-parts(8)> tool. L<run-parts(8)> runs
all the executable files named within constraints described below,
found in the given directory.  Other files and directories are
silently ignored.

Additionally it can just print the names of the all matching files
(not limited to executables, but ignores blacklisted files like
e.g. backup files), but don't actually run them.

This is useful when functionality or configuration is split over
multiple files in one directory.

This module is not thought to be used directly and its interface may
change. See L<Run::Parts> for a stable user interface.

=head1 METHODS

=head2 new (Constructor)

Creates a new L<Run::Parts> object. Takes one parameter, the directory on
which run-parts should work.

=head2 run_parts_command

Returns the L<run-parts(8)> command to run with the given command
parameter

=head1 SEE ALSO

L<Run::Parts>, L<run-parts(8)>

=head1 BUGS

Please report any bugs or feature requests to C<bug-run-parts at
rt.cpan.org>, or through the web interface at
L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Run-Parts>.  I will
be notified, and then you'll automatically be notified of progress on
your bug as I make changes.

=head1 AUTHOR

Axel Beckert <abe@deuxchevaux.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by Axel Beckert.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
