#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2023 -- leonerd@leonerd.org.uk

use v5.36;
use Object::Pad 0.66;

class App::perl::distrolint::Check::Unimport
   :does(App::perl::distrolint::CheckRole::EachFile)
   :does(App::perl::distrolint::CheckRole::TreeSitterPerl);

use Text::Treesitter 0.07; # child_by_field_name

use constant DESC => "check that if import exists then so does unimport";
use constant SORT => 20;

=head1 NAME

C<App::perl::distrolint::Check::Unimport> - check that a C<sub unimport> is defined if required

=head1 DESCRIPTION

This checks that if any package defines a C<sub import>, then it also defines
a C<sub unimport>. This helps to ensure that whatever effects the module
normally provides at import time can be undone by the C<no ...> syntax.
=cut

method run ( $app )
{
   return $self->run_for_each_perl_file( check_file => $app );
}

method check_file ( $file, $app )
{
   my $tree = $self->parse_perl_file( $file );

   my $have_import;
   my $have_unimport;

   STMT: foreach my $node ( $tree->root_node->child_nodes ) {
      next if $node->is_extra;
      next unless $node->is_named and $node->type eq "subroutine_declaration_statement";

      my $subname = $node->child_by_field_name( "name" )->text;

      $have_import++   if $subname eq "import";
      $have_unimport++ if $subname eq "unimport";
   }

   return 1 if $have_unimport or !$have_import;

   $app->diag( "%s has sub import but no sub unimport", $file );
   return 0;
}

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>

=cut

0x55AA;
